import sys              # used for file reading
import time             # used for timing the path-finding
from heapq import *     # used for implementing the A* open list
from settings import *  # use a separate file for all the constant settings
import math

# the class we will use to store the map, and make calls to path finding
class Grid:
    # set up all the default values for the frid and read in the map from a given file
    def __init__(self, filename):
        # 2D list that will hold all of the grid tile information 
        self.__grid = []
        self.__load_data(filename)
        self.__width, self.__height = len(self.__grid), len(self.__grid[0])
        # compute the connectivity sectors for determining path walkability
        self.__compute_connectivity()
        # set up the AStar object we'll use to call for path finding
        # the constructor does some pre-computation to make individual queries faster
        self.__astar = AStar(self)
        self.__legal_actions = {}
        for x in range(self.width()):
            for y in range(self.height()):
                for s in range(1, MAX_SIZE+1):   # for each legal object size
                    self.__legal_actions[(x,y,s)] = []
                    for action in LEGAL_ACTIONS:    # test each of the ocstate actions for legality
                        if self.is_legal_action((x,y), action, s):
                            self.__legal_actions[(x,y,s)].append((action, (x+action[0], y+action[1]), self.get_action_cost(action)))

    # loads the grid data from a given file name
    def __load_data(self, filename):
        # turn each line in the map file into a list of integers
        temp_grid = [list(map(int,line.strip())) for line in open(filename, 'r')]
        # transpose the input since we read it in as (y, x) 
        self.__grid = [list(i) for i in zip(*temp_grid)]

    def get_legal_actions(self, tile, size):
        return self.__legal_actions[(tile[0], tile[1], size)]

    # computes the walkable sectors on the map via a 4-directional flood-fill
    # 4-directions is sufficient since diagonal actionments can't jump over corners
    # sectors is a 3D list that will hold all of the tile connectivity information
    # sectors[size][x1][y1] = integer denoting the sector a tile is in
    # tiles are connected iff sectors[size][x1][y1] == sectors[size][x2][y2]
    # sectors[size][x1][y1] == 0 means that size cannot fit on that tile location
    def __compute_connectivity(self):
        self.__sectors = []
        self.__num_sectors = [0] * (MAX_SIZE+1)
        self.__sectors.append([[0]*self.height() for x in range(self.width())])
        for s in range(1, MAX_SIZE+1):
            self.__sectors.append([[0]*self.height() for x in range(self.width())])
            for x in range(self.width()):
                for y in range(self.height()):
                    self.__num_sectors[s] = self.__num_sectors[s] + 1
                    self.__flood_fill((x,y), self.get((x,y)), s, self.__num_sectors[s])

    # flood fill out in 4 directions from a given tile
    def __flood_fill(self, tile, val, size, sector):
        # if we are out of bounds, stop
        if self.__is_oob(tile, size): return 
        # if we have already assigned a sector for this tile, stop
        if self.__get_sector(tile, size) != 0: return 
        # if the tile is not the same terrain, mark it bad and stop
        if not self.__check_terrain(tile, size, val): return 
        # we know the tile is valid at this point so add it to the sector
        self.__sectors[size][tile[0]][tile[1]] = self.__num_sectors[size]
        # generate all tiles around this one and recurse
        self.__flood_fill((tile[0],   tile[1]+1), val, size, sector)
        self.__flood_fill((tile[0],   tile[1]-1), val, size, sector)
        self.__flood_fill((tile[0]+1,   tile[1]), val, size, sector)
        self.__flood_fill((tile[0]-1,   tile[1]), val, size, sector)

    # check whether we can make a action from a given tile
    def is_legal_action(self, start, action, size):
        # check if the action will place us out of bounds
        tile = (start[0] + action[0], start[1] + action[1])
        if tile[0] < 0 or tile[1] < 0 or tile[0] + size > self.width() or tile[1] + size > self.height(): return False
        # check to see if the action places us in the same sector
        if not self.is_connected(start, tile, size): return False
        # finally, check to see if we are not 'jumping over' corner tiles with a diagonal action
        return self.is_connected(start, (tile[0], start[1]), size) and self.is_connected(start, (start[0], tile[1]), size)

    # checks whether a tile and size goes out of bounds on the map
    def __is_oob(self, tile, size):
        return tile[0] < 0 or tile[1] < 0 or tile[0]+size > self.width() or tile[1]+size > self.height()

    # checks if a given tile contains only a given value for a given size
    def __check_terrain(self, tile, size, val):
        for sx in range(size):
            for sy in range(size):
                if self.__is_oob((sx,sy), size): return False
                if self.get((tile[0]+sx, tile[1]+sy)) != val: return False
        return True

    # checks if start, end tile are walkable with a given size
    # two tiles can be navigated between iff their sectors are equal (flood fill reaches)
    def is_connected(self, start, goal, size):
        return (self.__get_sector(start, size) == self.__get_sector(goal, size)) and (self.__get_sector(start, size) != 0)

    # returns the sector number that the tile belongs to at a given size
    def __get_sector(self, tile, size):
        return self.__sectors[size][tile[0]][tile[1]]

    # returns a sample path from start tile to end tile which is probably illegal
    def get_path_bird(self, start, end, size):
        path = []
        action = (1 if start[0] <= end[0] else -1, 1 if start[1] <= end[1] else -1)
        d = (abs(start[0] - end[0]), abs(start[1] - end[1]))
        # add the diagonal actions until we hit the row or column of the end tile
        for diag in range(d[1] if d[0] > d[1] else d[0]):
            path.append(action)
        # add the remaining straight actions to reach the end tile
        for straight in range(d[0]-d[1] if d[0] > d[1] else d[1]-d[0]):
            path.append((action[0], 0) if d[0]>d[1] else (0, action[1]))
        return path

    # returns the shortest path between start and goal tiles with a given size
    def get_path(self, start, goal, size):
        return self.__astar.get_path(start, goal, size)

    # estimate the cost for moving between start and end
    def estimate_cost(self, start, goal):
        d = [abs(start[0] - goal[0]), abs(start[1] - goal[1])]
        return (d[1] * DIAGONAL_COST + (d[0]-d[1]) * CARDINAL_COST) if (d[0] > d[1]) else (d[0] * DIAGONAL_COST + (d[1]-d[0]) * CARDINAL_COST)
        
    # return the cost of a given action
    def get_action_cost(self, action):
        return CARDINAL_COST if (action[0] == 0 or action[1] == 0) else DIAGONAL_COST 

    # returns the tile type of a given position
    def get(self, tile): return self.__grid[tile[0]][tile[1]]
    def width(self):     return self.__width
    def height(self):    return self.__height

class AStar:
    # constructor for AStar, takes in a grid based map
    def __init__(self, grid):
        self.__grid = grid
        self.__open_num = 0

    # add a node to the open list, which is a list / heap in this implementation
    def __add_to_open(self, open, node):
        heappush(open, (node.f, -node.g, node))    # push the node on the heap based on f-value
        self.__open_num += 1

    # get the min node from the open list
    def __pop_min_open(self, open):
        return heappop(open)[2]

    # reconstruct the path taken from a given node back to the start
    def __reconstruct_path(self, node):
        path = []
        # walk from the goal back to the start adding actions as we go
        while node.parent != None:
            path.append(node.action)
            node = node.parent
        # return the reverse of this path, since it was constructed backwards
        return path[::-1]

    # do the actual astar pathfinding
    def __get_path(self, start, goal, size):
        # if a path is not possible, then don't bother searching for one
        if not self.__grid.is_connected(start, goal, size): return [], 0, set()
        open, closed, all_nodes = [], set(), {}
        start_node = Node(start, None, None, 0, self.__grid.estimate_cost(start, goal))
        self.__add_to_open(open, start_node)
        all_nodes[start] = start_node
        while open:
            node = self.__pop_min_open(open)
            if (node.state in closed): continue
            closed.add(node.state)
            if node.state == goal: return self.__reconstruct_path(node), node.g, closed
            for action, child_state, action_cost in self.__grid.get_legal_actions(node.state, size):
                new_g = node.g + action_cost
                if child_state in all_nodes and all_nodes[child_state].g <= new_g: continue
                child = Node(child_state, node, action, new_g, self.__grid.estimate_cost(child_state, goal))
                self.__add_to_open(open, child)
                all_nodes[child.state] = child
        # we did not find a valid path, so these return values will be empty
        return [], 0, set()

    # do the astar pathfinding from start to goal with a given size
    def get_path(self, start, goal, size):
        return self.__get_path(start, goal, size)

# Node class that we'll use for A* search
class Node:
    # initialize a new node with 0 values
    def __init__(self, state, parent, action, g, h):
        self.state = state
        self.action = action
        self.parent = parent
        self.g = g
        self.f = g + h

    # less than operator, necessary for putting nodes into a heap in A*
    def __lt__(self, other):
        # tie break on g value
        if self.f == other.f:
            return self.g > other.g
        # compare f values for storing in the open list heap
        return self.f < other.f
